<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PostJobRequest;
use Yajra\DataTables\DataTables;
use App\Models\PostJobBid;
use App\Traits\NotificationTrait;
use Illuminate\Support\Facades\Log;

class PostJobRequestController extends Controller
{
    use NotificationTrait;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $filter = [
            'status' => $request->status,
        ];
        $pageTitle = trans('messages.job_request_list');
        $auth_user = authSession();
        $assets = ['datatable'];

        return view('postrequest.index', compact('pageTitle','auth_user','assets','filter'));

    }



    public function index_data(DataTables $datatable,Request $request)
    {
        $query = PostJobRequest::query();
        $filter = $request->filter;

        if (isset($filter)) {
            if (isset($filter['column_status'])) {
                $query->where('status', $filter['column_status']);
            }
        }
        if (auth()->user()->hasAnyRole(['admin'])) {
            $query->newQuery();
        }

        return $datatable->eloquent($query)
            ->addColumn('check', function ($row) {
                return '<input type="checkbox" class="form-check-input select-table-row"  id="datatable-row-'.$row->id.'"  name="datatable_ids[]" value="'.$row->id.'" onclick="dataTableRowCheck('.$row->id.')">';
            })
            ->editColumn('title', function($query){
                return '<a class="btn-link btn-link-hover"  href='.route('postjobrequest.service',$query->id).'>'.$query->title.'</a>';
            })

        //     ->editColumn('title', function($query) {
        //     // Check if the user has the 'post job request edit' permission
        //     if (auth()->user()->can('postjobrequest edit')) {
        //         // Create a link for the job request editing
        //         $link = '<a class="btn-link btn-link-hover" href=' . route('postjobrequest.service', $query->id) . '>' . $query->title . '</a>';
        //     } else {
        //         $link = $query->title;
        //     }

        //     // Add the image column for job request image (assuming 'service_attachment' is the media collection)
        //     $imageUrl = $query->getFirstMedia('service_attachment');

        //     // Check if an image exists
        //     if ($imageUrl) {
        //         $imageTag = '<img src="' . $imageUrl->getUrl() . '" alt="job request image" class="avatar avatar-40 rounded-pill mr-2">';
        //     } else {
        //         $imageTag = ''; // No image found
        //     }

        //     // Return both the image and title with flexbox for alignment
        //     return '<div style="display: flex; align-items: center;">' .
        //             $imageTag .
        //             '<span style="margin-left: 10px;">' . $link . '</span>' .
        //         '</div>';
        // })


            ->editColumn('provider_id' , function ($query){
                return view('postrequest.provider', compact('query'));
            })
            ->editColumn('customer_id' , function ($query){
                return view('postrequest.customer', compact('query'));
            })
            ->filterColumn('customer_id',function($query,$keyword){
                $query->whereHas('customer',function ($q) use($keyword){
                    $q->where('display_name','like','%'.$keyword.'%');
                });
            })
            ->orderColumn('customer_id', function ($query, $order) {
                $query->select('post_job_requests.*')
                      ->join('users as customers', 'customers.id', '=', 'post_job_requests.customer_id')
                      ->orderBy('customers.display_name', $order);
            })
            ->editColumn('price' , function ($query){
                return getPriceFormat($query->price);
            })
            ->editColumn('status' , function ($query){
               $status = $query->status;
                if($status == 'requested'){
                    $status = '<span class="badge text-primary bg-primary-subtle">'.__('messages.requested').'</span>';
                }
                return  $status;
            })

            ->addColumn('action', function($post_job){
                return view('postrequest.action',compact('post_job'))->render();
            })
            ->addIndexColumn()
            ->rawColumns(['title','action','status','check'])
            ->toJson();
    }

    /* bulck action method */
    public function bulk_action(Request $request)
    {
        $ids = explode(',', $request->rowIds);

        $actionType = $request->action_type;

        $message = 'Bulk Action Updated';
        switch ($actionType) {
            case 'change-status':
                $branches = PostJobRequest::whereIn('id', $ids)->update(['status' => $request->status]);
                $message = 'Bulk PostJobRequest Status Updated';
                break;

            case 'delete':
                PostJobRequest::whereIn('id', $ids)->delete();
                $message = 'Bulk PostJobRequest Deleted';
                break;

            default:
                return response()->json(['status' => false, 'message' => 'Action Invalid']);
                break;
        }

        return response()->json(['status' => true, 'message' => $message]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $data = $request->all();
        $data['customer_id'] =  !empty($request->customer_id) ? $request->customer_id : auth()->user()->id;

        $result = PostJobRequest::updateOrCreate(['id' => $request->id], $data);
        $activity_data = [
            'activity_type' => 'job_requested',
            'post_job_id' => $result->id,
            'post_job' => $result,
            'latitude' =>isset($data['latitude']) ? $data['latitude'] : 0.0,
            'longitude' => isset($data['longitude']) ? $data['longitude'] : 0.0,
        ];
        $this->sendNotification($activity_data);

         if($result->postServiceMapping()->count() > 0)
        {
            $result->postServiceMapping()->delete();
        }
        if ($request->has('service_id')) {
            if (is_array($request->service_id)) {
                foreach ($request->service_id as $service) {
                    $post_services = [
                        'post_request_id' => $result->id,
                        'service_id' => $service,
                    ];
                    $result->postServiceMapping()->insert($post_services);
                }
            }
        }
        if($request->status == 'assigned'){
            $activity_data = [
                'activity_type' => 'user_accept_bid',
                'post_job_id' => $result->id,
                'post_job' => $result,
                'customer_name' => optional($result->customer)->display_name ?? null,
                'provider_name' => optional($result->provider)->display_name ?? null,
                'latitude' =>isset($data['latitude']) ? $data['latitude'] : 0.0,
                'longitude' => isset($data['longitude']) ? $data['longitude'] : 0.0,
            ];
            $this->sendNotification($activity_data);

        }
        $message = __('messages.update_form',[ 'form' => __('messages.postrequest') ] );
		if($result->wasRecentlyCreated){
			$message = __('messages.save_form',[ 'form' => __('messages.postrequest') ] );
		}

        if($request->is('api/*')) {
            return comman_message_response($message);
		}

		return redirect(route('service.index'))->withSuccess($message);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $pageTitle = trans('messages.list_form_title',['form' => trans('messages.postbid')] );
        $auth_user = authSession();
        $asset = ['datatable'];
        return view('postrequest.view', compact('pageTitle', 'auth_user', 'asset','id'));
    }

    public function postrequest_index_data(DataTables $datatable,$id)
    {
        $query = PostJobBid::where('post_request_id',$id);

        if (auth()->user()->hasAnyRole(['admin'])) {
            $query->newquery();
        }

        return $datatable  ->eloquent($query)
        ->editColumn('post_request_id' , function ($post_job_bid){
            return ($post_job_bid->post_request_id != null && isset($post_job_bid->postrequest)) ? $post_job_bid->postrequest->title : '-';
        })
        ->editColumn('provider_id' , function ($post_job_bid){
            return ($post_job_bid->provider_id != null && isset($post_job_bid->provider)) ? $post_job_bid->provider->display_name : '-';
        })
        ->editColumn('customer_id', function ($post_job_bid){
            return ($post_job_bid->customer_id != null && isset($post_job_bid->customer)) ? $post_job_bid->customer->display_name : '-';
        })
        ->editColumn('price' , function ($post_job){
            return getPriceFormat($post_job->price);
        })
        ->editColumn('duration' , function ($post_job_bid){
            return ($post_job_bid->duration != null) ? $post_job_bid->duration : '-';
        })
        ->addIndexColumn()
        ->toJson();
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        if(demoUserPermission()){
            if(request()->is('api/*')){
                return comman_message_response( __('messages.demo_permission_denied') );
            }
            return  redirect()->back()->withErrors(trans('messages.demo_permission_denied'));
        }
        $post_request = PostJobRequest::find($id);
        //$post_request->delete();
        $msg= __('messages.msg_fail_to_delete',['item' => __('messages.postrequest')] );

        if($post_request!='') {
            if($post_request->postServiceMapping()->count() > 0)
            {
                $post_request->postServiceMapping()->delete();
            }
            if($post_request->postBidList()->count() > 0)
            {
                $post_request->postBidList()->delete();
            }
            $post_request->delete();
            $msg= __('messages.msg_deleted',['name' => __('messages.postrequest')] );
        }
        if(request()->is('api/*')){
            return comman_custom_response(['message'=> $msg , 'status' => true]);
        }
        return redirect()->back()->withSuccess($msg);

    }
}
